<?php
/**
 * Shipping Import & Export class.
 *
 * @package WooCommerce_Table_Rate_Shipping
 */

namespace WooCommerce\Shipping\Table_Rate;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class for handling import/export of shipping instance settings.
 */
class Shipping_Rates_Import_Export {

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'admin_post_export_shipping_instance_settings', array( $this, 'export_settings' ) );
		add_action( 'admin_init', array( $this, 'register_importer' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_styles' ) );
	}

	/**
	 * Handle export action and generate CSV file.
	 *
	 * @return void
	 */
	public function export_settings() {
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_die( esc_html__( 'Unauthorized.', 'woocommerce-table-rate-shipping' ) );
		}

		$instance_id = isset( $_GET['instance_id'] ) ? absint( $_GET['instance_id'] ) : 0;

		if ( ! $instance_id || ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( wc_clean( wp_unslash( $_GET['_wpnonce'] ) ), 'export_instance_' . $instance_id ) ) {
			wp_die( esc_html__( 'Invalid nonce verification.', 'woocommerce-table-rate-shipping' ) );
		}

		global $wpdb;

		$table_rates = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$wpdb->prefix}woocommerce_shipping_table_rates WHERE shipping_method_id = %d", $instance_id ), ARRAY_A );

		if ( empty( $table_rates ) ) {
			wp_die( esc_html__( 'No table rates found for this instance.', 'woocommerce-table-rate-shipping' ) );
		}

		header( 'Content-Type: text/csv; charset=utf-8' );
		header( 'Content-Disposition: attachment; filename=table-rates-instance-' . $instance_id . '-' . gmdate( 'Y-m-d' ) . '.csv' );

		$output = fopen( 'php://output', 'w' );

		fputcsv( $output, array_keys( $table_rates[0] ) );

		foreach ( $table_rates as $rate ) {
			fputcsv( $output, $rate );
		}

		fclose( $output );
		exit;
	}

	/**
	 * Register WordPress importer.
	 *
	 * @return void
	 */
	public function register_importer() {
		if ( ! defined( 'WP_LOAD_IMPORTERS' ) ) {
			return;
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( ! isset( $_GET['import'] ) || 'wc_table_rates' !== $_GET['import'] ) {
			return;
		}

		register_importer(
			'wc_table_rates',
			esc_html__( 'WooCommerce Table Rates', 'woocommerce-table-rate-shipping' ),
			esc_html__( 'Import WooCommerce Table Rates.', 'woocommerce-table-rate-shipping' ),
			array( $this, 'run_importer' )
		);
	}

	/**
	 * Run importer.
	 *
	 * @return void
	 */
	public function run_importer() {
		echo '<div class="wrap woocommerce">';
		echo '<h1>' . esc_html__( 'Import WooCommerce shipping settings', 'woocommerce-table-rate-shipping' ) . '</h1>';
		$importer = new Shipping_Rates_Importer();
		$importer->dispatch();
		echo '</div>';
	}

	/**
	 * Enqueue necessary admin styles.
	 */
	public function enqueue_styles() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( isset( $_GET['import'] ) && 'wc_table_rates' === $_GET['import'] ) {
			wp_enqueue_style( 'woocommerce_admin_styles', WC()->plugin_url() . '/assets/css/admin.css', array(), WC_VERSION );
		}
	}
}

new Shipping_Rates_Import_Export();
