<?php
/**
 * Shipping Settings Importer class.
 *
 * @package WooCommerce_Table_Rate_Shipping
 */

namespace WooCommerce\Shipping\Table_Rate;

if ( ! class_exists( 'WP_Importer' ) ) {
	require ABSPATH . 'wp-admin/includes/class-wp-importer.php';
}

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use WP_Importer;

/**
 * Class for importing table rates from CSV files.
 */
class Shipping_Rates_Importer extends WP_Importer {

	/**
	 * Instance ID.
	 *
	 * @var int
	 */
	private int $instance_id;

	/**
	 * Dispatches the importer based on the current step.
	 *
	 * @return void
	 */
	public function dispatch() {
		$step              = empty( $_GET['step'] ) ? 0 : absint( $_GET['step'] );
		$this->instance_id = isset( $_GET['instance_id'] ) ? intval( $_GET['instance_id'] ) : false;

		if ( ! $this->instance_id ) {
			wp_die( esc_html__( 'The instance ID is required to process the import.', 'woocommerce-table-rate-shipping' ) );
		}

		switch ( $step ) {
			case 0:
				$this->upload_form();
				break;
			case 1:
				check_admin_referer( 'import-upload' );
				$file = wp_import_handle_upload();

				if ( isset( $file['error'] ) ) {
					wp_die( esc_html( $file['error'] ) );
				}

				$override_table_rates = isset( $_POST['override_table_rates'] ) && 'yes' === $_POST['override_table_rates'];

				$this->import( $file['file'], $override_table_rates );
				break;
		}
	}

	/**
	 * Imports shipping rates from a CSV file.
	 *
	 * @param string $file_path Path to uploaded CSV file.
	 * @param bool   $override_table_rates Whether to override existing table rates.
	 */
	private function import( string $file_path, bool $override_table_rates ) {
		global $wpdb;

		if ( ! file_exists( $file_path ) || ! is_readable( $file_path ) ) {
			wp_die( esc_html__( 'Cannot read the uploaded file.', 'woocommerce-table-rate-shipping' ) );
		}

		$instance_id = $this->instance_id;

		// Parse CSV file.
		$handle = fopen( $file_path, 'r' );
		if ( false === $handle ) {
			wp_die( esc_html__( 'Error opening the CSV file.', 'woocommerce-table-rate-shipping' ) );
		}

		$header = fgetcsv( $handle );
		if ( ! $header ) {
			wp_die( esc_html__( 'CSV file appears to be empty or invalid.', 'woocommerce-table-rate-shipping' ) );
		}

		$table_rates = array();

		while ( ( $row = fgetcsv( $handle ) ) !== false ) {
			$rate = array_combine( $header, $row );
			if ( ! empty( $rate ) ) {
				unset( $rate['rate_id'] ); // remove rate_id if exists to prevent conflicts.
				$rate['shipping_method_id'] = $instance_id;
				$table_rates[]              = $rate;
			}
		}

		fclose( $handle );

		if ( empty( $table_rates ) ) {
			wp_die( esc_html__( 'No valid table rates found in the CSV file.', 'woocommerce-table-rate-shipping' ) );
		}

		// Override existing table rates if requested.
		if ( $override_table_rates ) {
			$wpdb->delete(
				"{$wpdb->prefix}woocommerce_shipping_table_rates",
				array( 'shipping_method_id' => $instance_id ),
				array( '%d' )
			);
		}

		// Insert new table rates.
		foreach ( $table_rates as $rate ) {
			$wpdb->insert( "{$wpdb->prefix}woocommerce_shipping_table_rates", $rate );
		}

		echo '<div class="notice notice-success"><p>' .
			esc_html__( 'Shipping table rates imported successfully.', 'woocommerce-table-rate-shipping' ) .
			' <a href="' . esc_url( admin_url( 'admin.php?page=wc-settings&tab=shipping&instance_id=' . $instance_id ) ) . '" class="button">' .
			esc_html__( 'Go back to instance settings', 'woocommerce-table-rate-shipping' ) .
			'</a></p></div>';
	}

	/**
	 * Displays the upload form for importing CSV file.
	 *
	 * @return void
	 */
	private function upload_form() {
		?>
		<div class="woocommerce-progress-form-wrapper">
			<form class="wc-progress-form-content woocommerce-importer" enctype="multipart/form-data" method="post" action="<?php echo esc_url( add_query_arg( 'step', 1 ) ); ?>">
				<header>
					<h2><?php esc_html_e( 'Import WooCommerce shipping rates from a CSV file.', 'woocommerce-table-rate-shipping' ); ?></h2>
					<p><?php esc_html_e( 'This tool allows you to import WooCommerce Table Rates entries.', 'woocommerce-table-rate-shipping' ); ?></p>
				</header>
				<section>
					<table class="form-table woocommerce-importer-options">
						<tbody>
						<tr>
							<th scope="row">
								<label for="upload"><?php esc_html_e( 'Choose a CSV file from your computer:', 'woocommerce-table-rate-shipping' ); ?></label>
							</th>
							<td>
								<input type="file" id="upload" name="import" required accept=".csv">
								<input type="hidden" name="action" value="save">
								<br><small><?php esc_html_e( 'Maximum size:', 'woocommerce-table-rate-shipping' ); ?> <?php echo esc_html( size_format( wp_max_upload_size() ) ); ?></small>
							</td>
						</tr>
						<tr>
							<th>
								<label for="override_table_rates"><?php esc_html_e( 'Override Table Rates entries?', 'woocommerce-table-rate-shipping' ); ?></label>
							</th>
							<td>
								<input type="checkbox" id="override_table_rates" name="override_table_rates" value="yes">
								<label for="override_table_rates"><?php esc_html_e( 'Replace existing Table Rates with imported data.', 'woocommerce-table-rate-shipping' ); ?></label>
							</td>
						</tr>
						</tbody>
					</table>
				</section>
				<div class="wc-actions">
					<button type="submit" class="button button-primary button-next" value="Continue" name="save_step">
						<?php esc_html_e( 'Continue', 'woocommerce-table-rate-shipping' ); ?>
					</button>
					<?php wp_nonce_field( 'import-upload' ); ?>
				</div>
			</form>
		</div>
		<?php
	}
}
